<?php
namespace ElementorCounter\Widgets;

use \Elementor\Controls_Manager as Controls_Manager;
use \Elementor\Frontend;
use \Elementor\Group_Control_Border as Group_Control_Border;
use \Elementor\Group_Control_Box_Shadow as Group_Control_Box_Shadow;
use \Elementor\Group_Control_Typography as Group_Control_Typography;
use \Elementor\Utils as Utils;
use \Elementor\Widget_Base as Widget_Base;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Elementor Counter
 *
 * Elementor widget for team vision
 *
 * @since 1.0.0
 */
class Counter extends Widget_Base {

	/**
	 * Retrieve the widget name.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'simple-counter';
	}

	/**
	 * Retrieve the widget title.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title() {
		return esc_html__( 'Simple Counter', 'elementor-counter' );
	}

	/**
	 * Retrieve the widget icon.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'eicon-counter';
	}

	/**
	 * Retrieve the list of categories the widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * Note that currently Elementor supports only one category.
	 * When multiple categories passed, Elementor uses the first one.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'general' ];
	}

	/**
	 * Retrieve the list of scripts the widget depended on.
	 *
	 * Used to set scripts dependencies required to run the widget.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return array Widget scripts dependencies.
	 */
	public function get_script_depends() {
		return [ 'elementor-counter' ];
	}

	/**
	 * Register the widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function _register_controls() {

		$this->start_controls_section(
			'section_content',
			[
				'label' => esc_html__( 'Content', 'elementor-counter' ),
			]
		);

		$this->add_control(
			'number',
			[
				'label' => esc_html__( 'Number', 'elementor-counter' ),
				'type' => Controls_Manager::TEXT,
				'default' => '10'
			]
		);

		$this->add_control(
			'speed',
			[
				'label' => esc_html__( 'Speed (ms)', 'elementor-counter' ),
				'type' => Controls_Manager::TEXT,
				'default' => '10000'
			]
		);
		
		$this->end_controls_section();

		$this->start_controls_section(
			'section_animation',
			[
				'label' => esc_html__( 'Animations', 'elementor-counter' )
			]
		);
		
		$this->add_control(
			'addon_animate',
			[
				'label' => esc_html__( 'Animate', 'elementor-counter' ),
				'type' => Controls_Manager::SELECT,
				'default' => 'off',
				'options' => [
					'off'	=> 'Off',
					'on' 	=> 'On'					
				]
			]
		);		

		$this->add_control(
			'effect',
			[
				'label' => esc_html__( 'Animate Effects', 'elementor-counter' ),
				'type' => Controls_Manager::SELECT,
				'default' => 'fade-in',
				'options' => [
							'fade-in'			=> esc_html__( 'Fade In', 'elementor-counter' ),
							'fade-in-up' 		=> esc_html__( 'fade in up', 'elementor-counter' ),					
							'fade-in-down' 		=> esc_html__( 'fade in down', 'elementor-counter' ),					
							'fade-in-left' 		=> esc_html__( 'fade in Left', 'elementor-counter' ),					
							'fade-in-right' 	=> esc_html__( 'fade in Right', 'elementor-counter' ),					
							'fade-out'			=> esc_html__( 'Fade In', 'elementor-counter' ),
							'fade-out-up' 		=> esc_html__( 'Fade Out up', 'elementor-counter' ),					
							'fade-out-down' 	=> esc_html__( 'Fade Out down', 'elementor-counter' ),					
							'fade-out-left' 	=> esc_html__( 'Fade Out Left', 'elementor-counter' ),					
							'fade-out-right' 	=> esc_html__( 'Fade Out Right', 'elementor-counter' ),
							'bounce-in'			=> esc_html__( 'Bounce In', 'elementor-counter' ),
							'bounce-in-up' 		=> esc_html__( 'Bounce in up', 'elementor-counter' ),					
							'bounce-in-down' 	=> esc_html__( 'Bounce in down', 'elementor-counter' ),					
							'bounce-in-left' 	=> esc_html__( 'Bounce in Left', 'elementor-counter' ),					
							'bounce-in-right' 	=> esc_html__( 'Bounce in Right', 'elementor-counter' ),					
							'bounce-out'		=> esc_html__( 'Bounce In', 'elementor-counter' ),
							'bounce-out-up' 	=> esc_html__( 'Bounce Out up', 'elementor-counter' ),					
							'bounce-out-down' 	=> esc_html__( 'Bounce Out down', 'elementor-counter' ),					
							'bounce-out-left' 	=> esc_html__( 'Bounce Out Left', 'elementor-counter' ),					
							'bounce-out-right' 	=> esc_html__( 'Bounce Out Right', 'elementor-counter' ),	
							'zoom-in'			=> esc_html__( 'Zoom In', 'elementor-counter' ),
							'zoom-in-up' 		=> esc_html__( 'Zoom in up', 'elementor-counter' ),					
							'zoom-in-down' 		=> esc_html__( 'Zoom in down', 'elementor-counter' ),					
							'zoom-in-left' 		=> esc_html__( 'Zoom in Left', 'elementor-counter' ),					
							'zoom-in-right' 	=> esc_html__( 'Zoom in Right', 'elementor-counter' ),					
							'zoom-out'			=> esc_html__( 'Zoom In', 'elementor-counter' ),
							'zoom-out-up' 		=> esc_html__( 'Zoom Out up', 'elementor-counter' ),					
							'zoom-out-down' 	=> esc_html__( 'Zoom Out down', 'elementor-counter' ),					
							'zoom-out-left' 	=> esc_html__( 'Zoom Out Left', 'elementor-counter' ),					
							'zoom-out-right' 	=> esc_html__( 'Zoom Out Right', 'elementor-counter' ),
							'flash' 			=> esc_html__( 'Flash', 'elementor-counter' ),
							'strobe'			=> esc_html__( 'Strobe', 'elementor-counter' ),
							'shake-x'			=> esc_html__( 'Shake X', 'elementor-counter' ),
							'shake-y'			=> esc_html__( 'Shake Y', 'elementor-counter' ),
							'bounce' 			=> esc_html__( 'Bounce', 'elementor-counter' ),
							'tada'				=> esc_html__( 'Tada', 'elementor-counter' ),
							'rubber-band'		=> esc_html__( 'Rubber Band', 'elementor-counter' ),
							'swing' 			=> esc_html__( 'Swing', 'elementor-counter' ),
							'spin'				=> esc_html__( 'Spin', 'elementor-counter' ),
							'spin-reverse'		=> esc_html__( 'Spin Reverse', 'elementor-counter' ),
							'slingshot'			=> esc_html__( 'Slingshot', 'elementor-counter' ),
							'slingshot-reverse'	=> esc_html__( 'Slingshot Reverse', 'elementor-counter' ),
							'wobble'			=> esc_html__( 'Wobble', 'elementor-counter' ),
							'pulse' 			=> esc_html__( 'Pulse', 'elementor-counter' ),
							'pulsate'			=> esc_html__( 'Pulsate', 'elementor-counter' ),
							'heartbeat'			=> esc_html__( 'Heartbeat', 'elementor-counter' ),
							'panic' 			=> esc_html__( 'Panic', 'elementor-counter' )				
				],
				'condition'	=> [
					'addon_animate'	=> 'on'
				]
			]
		);			

		$this->add_control(
			'delay',
			[
				'label' => esc_html__( 'Animate Delay (ms)', 'elementor-counter' ),
				'type' => Controls_Manager::TEXT,
				'default' => '1000',
				'condition'	=> [
					'addon_animate'	=> 'on'
				]
			]
		);	
		
		$this->end_controls_section();


		$this->start_controls_section(
			'section_style',
			[
				'label' => esc_html__( 'Style', 'elementor-counter' ),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'custom_style',
			[
				'label' => esc_html__( 'Custom Style', 'elementor-counter' ),
				'type' => Controls_Manager::SELECT,
				'default' => 'off',
				'options' => [
					'off'	=> 'Off',
					'on' 	=> 'On'					
				]
			]
		);

		$this->add_control(
			'color',
			[
				'label' => esc_html__( 'Color', 'elementor-counter' ),
				'type' => \Elementor\Controls_Manager::COLOR,
				'default' => '#FFFFFF',
				'condition'	=> [
					'custom_style'	=> 'on'
				]
			]
		);

		$this->add_control(
			'size',
			[
				'label' => esc_html__( 'Counter size (px)', 'elementor-counter' ),
				'type' => Controls_Manager::TEXT,
				'default' => '30px',
				'condition'	=> [
					'custom_style'	=> 'on'
				]
			]
		);	

		$this->add_control(
			'weight',
			[
				'label' => esc_html__( 'Counter Weight', 'elementor-counter' ),
				'type' => Controls_Manager::SELECT,
				'default' => 'normal',
				'options' => [
					'normal'	=> esc_html__( 'Normal', 'elementor-counter' ),
					'bold' 		=> esc_html__( 'Bold', 'elementor-counter' ),				
					'100' 		=> '100',				
					'200' 		=> '200',				
					'300' 		=> '300',				
					'400' 		=> '400',				
					'500' 		=> '500',				
					'600' 		=> '600',				
					'700' 		=> '700',				
					'800' 		=> '800',				
					'900' 		=> '900'				
				],
				'condition'	=> [
					'custom_style'	=> 'on'
				]
			]
		);

		$this->add_control(
			'padding',
			[
				'label' => esc_html__( 'Padding', 'elementor-counter' ),
				'type' => Controls_Manager::TEXT,
				'default' => '10px',
				'condition'	=> [
					'custom_style'	=> 'on'
				]
			]
		);	

		$this->add_control(
			'background_options',
			[
				'label' => esc_html__( 'Background Options', 'elementor-counter' ),
				'type' => Controls_Manager::SELECT,
				'default' => 'off',
				'options' => [
					'off'	=> 'Off',
					'on' 	=> 'On'					
				],
				'condition'	=> [
					'custom_style'	=> 'on'
				]
			]
		);

		$this->add_control(
			'background_color',
			[
				'label' => esc_html__( 'Background Color', 'elementor-counter' ),
				'type' => \Elementor\Controls_Manager::COLOR,
				'default' => '#000000',
				'condition'	=> [
					'background_options'	=> 'on'
				]
			]
		);

		$this->add_control(
			'border_radius',
			[
				'label' => esc_html__( 'Border Radius (ex: 10px or leave empty to border square)', 'elementor-counter' ),
				'type' => Controls_Manager::TEXT,
				'label_block' => true,
				'default' => '10px',
				'condition'	=> [
					'background_options'	=> 'on'
				]
			]
		);

		$this->end_controls_section();
		
	}

	 
	 /**
	 * Render the widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function render() {
		static $instance = 0;
		$instance++;		
		$settings = $this->get_settings_for_display();
		
        $number					= esc_html($settings['number']);		
        $speed					= esc_html($settings['speed']);		
		$custom_style			= esc_html($settings['custom_style']);
        $color					= esc_html($settings['color']);
        $size					= esc_html($settings['size']);
        $weight					= esc_html($settings['weight']);
		$padding				= esc_html($settings['padding']);
		$background_options		= esc_html($settings['background_options']);
		$background_color		= esc_html($settings['background_color']);
		$border_radius			= esc_html($settings['border_radius']);		
		
		// Animations
		$addon_animate			= esc_html($settings['addon_animate']);
		$effect					= esc_html($settings['effect']);
		$delay					= esc_html($settings['delay']);
		
		wp_enqueue_script( 'countTo' );
		wp_enqueue_style( 'animations' );
		wp_enqueue_script( 'appear' );			
		wp_enqueue_script( 'animate' );
		
		$css_inline = 'style="';
		
		// Custom Style
		if($custom_style == 'on') :
			$css_inline .= 'color:'.esc_html($color).';font-size:'.esc_html($size).';font-weight:'.esc_html($weight).';padding:'.esc_html($padding).';';
			if($background_options == 'on') :
				$css_inline .= 'background:'.esc_html($background_color).';border-radius:'.esc_html($border_radius).';';
			endif;
		endif;
		
		$css_inline .= '"';		
		
        echo '<div class="counter simple-counter-container '.counter_animate_class($addon_animate,$effect,$delay).'>';
		echo '<span class="simple-counter timer" '.$css_inline.' data-istance="simple-counter-'.esc_html($instance).'" data-to="'.esc_html($number).'" data-speed="'.esc_html($speed).'"></span>';
		echo '</div>';
		

		
	}

	/**
	 * Render the widget output in the editor.
	 *
	 * Written as a Backbone JavaScript template and used to generate the live preview.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function _content_template() {}
}
